<?php
/**
 * Copyright 2021 Adobe
 * All Rights Reserved.
 */

declare(strict_types=1);

namespace Magento\Framework\Webapi\Validator;

use Magento\Framework\Api\SearchCriteriaInterface;
use Magento\Framework\App\ObjectManager;
use Magento\Framework\Exception\InvalidArgumentException;
use Magento\Framework\Webapi\Validator\IOLimit\IOLimitConfigProvider;

/**
 * Validates search criteria inputs
 */
class SearchCriteriaValidator implements ServiceInputValidatorInterface
{
    /**
     * @var int
     */
    private $maximumPageSize;

    /**
     * @var IOLimitConfigProvider|null
     */
    private $configProvider;

    /**
     * @param int $maximumPageSize
     * @param IOLimitConfigProvider|null $configProvider
     */
    public function __construct(int $maximumPageSize, ?IOLimitConfigProvider $configProvider = null)
    {
        $this->maximumPageSize = $maximumPageSize;
        $this->configProvider = $configProvider ?? ObjectManager::getInstance()
            ->get(IOLimitConfigProvider::class);
    }

    /**
     * @inheritDoc
     * phpcs:disable Magento2.CodeAnalysis.EmptyBlock
     */
    public function validateComplexArrayType(string $className, array $items): void
    {
    }

    /**
     * @inheritDoc
     */
    public function validateEntityValue(object $entity, string $propertyName, $value): void
    {
        if ($entity instanceof SearchCriteriaInterface
            && $propertyName === 'pageSize'
            && $this->configProvider->isInputLimitingEnabled()
            && $value > ($max = $this->configProvider->getMaximumPageSize() ?? $this->maximumPageSize)
        ) {
            throw new InvalidArgumentException(
                __('Maximum SearchCriteria pageSize is %max', ['max' => $max])
            );
        }
    }
}
