<?php
/**
 *
 * Copyright 2015 Adobe
 * All Rights Reserved.
 */
namespace Magento\Catalog\Api;

use Magento\TestFramework\Helper\Bootstrap;
use Magento\TestFramework\TestCase\WebapiAbstract;

class CategoryLinkRepositoryTest extends WebapiAbstract
{
    const SERVICE_WRITE_NAME = 'catalogCategoryLinkRepositoryV1';
    const SERVICE_VERSION = 'V1';
    const RESOURCE_PATH_SUFFIX = '/V1/categories';
    const RESOURCE_PATH_PREFIX = 'products';

    private static $categoryId = 333;

    /**
     * @dataProvider saveDataProvider
     * @magentoApiDataFixture Magento/Catalog/_files/products_in_category.php
     * @param int $productId
     * @param string[] $productLink
     * @param int $productPosition
     */
    public function testSave($productLink, $productId, $productPosition = 0)
    {
        $serviceInfo = [
            'rest' => [
                'resourcePath' => self::RESOURCE_PATH_SUFFIX
                    . '/' . self::$categoryId . '/' . self::RESOURCE_PATH_PREFIX,
                'httpMethod' => \Magento\Framework\Webapi\Rest\Request::HTTP_METHOD_POST,
            ],
            'soap' => [
                'service' => self::SERVICE_WRITE_NAME,
                'serviceVersion' => self::SERVICE_VERSION,
                'operation' => self::SERVICE_WRITE_NAME . 'Save',
            ],
        ];
        $result = $this->_webApiCall($serviceInfo, ['productLink' => $productLink]);
        $this->assertTrue($result);
        $this->assertTrue($this->isProductInCategory(self::$categoryId, $productId, $productPosition));
    }

    public static function saveDataProvider()
    {
        return [
            [
                ['sku' => 'simple_with_cross', 'position' => 7, 'category_id' => self::$categoryId],
                334,
                7,
            ],
            [
                ['sku' => 'simple_with_cross', 'category_id' => self::$categoryId],
                334,
                0
            ],
        ];
    }

    /**
     * @dataProvider updateProductProvider
     * @magentoApiDataFixture Magento/Catalog/_files/products_in_category.php
     * @param int $productId
     * @param string[] $productLink
     * @param int $productPosition
     */
    public function testUpdateProduct($productLink, $productId, $productPosition = 0)
    {
        $serviceInfo = [
            'rest' => [
                'resourcePath' => self::RESOURCE_PATH_SUFFIX
                    . '/' . self::$categoryId . '/' . self::RESOURCE_PATH_PREFIX,
                'httpMethod' => \Magento\Framework\Webapi\Rest\Request::HTTP_METHOD_PUT,
            ],
            'soap' => [
                'service' => self::SERVICE_WRITE_NAME,
                'serviceVersion' => self::SERVICE_VERSION,
                'operation' => self::SERVICE_WRITE_NAME . 'Save',
            ],
        ];
        $result = $this->_webApiCall($serviceInfo, ['productLink' => $productLink]);
        $this->assertTrue($result);
        $this->assertFalse($this->isProductInCategory(self::$categoryId, $productId, $productPosition));
    }

    public static function updateProductProvider()
    {
        return [
            [
                ['sku' => 'simple_with_cross', 'position' => 7, 'categoryId' => self::$categoryId],
                333,
                4,
            ],
            [
                ['sku' => 'simple_with_cross', 'categoryId' => self::$categoryId],
                333,
                0
            ],
        ];
    }

    /**
     * @magentoApiDataFixture Magento/Catalog/_files/products_in_category.php
     */
    public function testDelete()
    {
        $serviceInfo = [
            'rest' => [
                'resourcePath' => self::RESOURCE_PATH_SUFFIX . '/' . self::$categoryId .
                    '/' . self::RESOURCE_PATH_PREFIX . '/simple',
                'httpMethod' => \Magento\Framework\Webapi\Rest\Request::HTTP_METHOD_DELETE,
            ],
            'soap' => [
                'service' => self::SERVICE_WRITE_NAME,
                'serviceVersion' => self::SERVICE_VERSION,
                'operation' => self::SERVICE_WRITE_NAME . 'DeleteByIds',
            ],
        ];
        $result = $this->_webApiCall(
            $serviceInfo,
            ['sku' => 'simple', 'categoryId' => self::$categoryId]
        );
        $this->assertTrue($result);
        $this->assertFalse($this->isProductInCategory(self::$categoryId, 333, 10));
    }

    /**
     * @param int $categoryId
     * @param int $productId
     * @param int $productPosition
     * @return bool
     */
    private function isProductInCategory($categoryId, $productId, $productPosition)
    {
        /** @var \Magento\Catalog\Api\CategoryRepositoryInterface $categoryLoader */
        $categoryLoader = Bootstrap::getObjectManager()
            ->create(\Magento\Catalog\Api\CategoryRepositoryInterface::class);
        $category = $categoryLoader->get($categoryId);
        $productsPosition = $category->getProductsPosition();

        if (isset($productsPosition[$productId]) && $productsPosition[$productId] == $productPosition) {
            return true;
        } else {
            return false;
        }
    }
}
